/**
* Script: image.js
* Written by: Andrew Helenius (Radnen)
* Updated: 8/21/2010
**/

/**
* Advanced Sphere Image Object: RadImage
*  - Zoom, rotate, and mask at the same time.
*    Mask get/set via a member object.
*    get/set the width and height.
*  - object (string): filename of image to load.
*  - object (Sphere Image): image associated w/ object.
**/
function Image(object)
{	
	// private data:
	var image = null;
	
	// either load from a file, or reference an object:
	if (typeof object == "string") image = LoadImage(object);
	else image = object;
	
	var sina = 1;
	var cosa = 1;
	var z = 1;
	var rotated = false;
	var image_w = image.width;
	var image_h = image.height;
	var img_w = image.width, img_h = image.height;
	var x1 = 0, x2 = image_w, x3 = image_w, x4 = 0;
	var y1 = 0, y2 = 0, y3 = image_h, y4 = image_h;
	var a = 0, b = 0;
	
	// public data:
	this.mask = CreateColor(255, 255, 255);
	this.__defineGetter__("width", function() { return image_w; });
	this.__defineGetter__("height", function() { return image_h; });
	this.__defineSetter__("width", function(v) { img_w = v * z; });
	this.__defineSetter__("height", function(v) { img_h = v * z; });
	
	/**
	* rotate(r);
	*  - rotates the image about its own axis by 'r' radians.
	**/
	this.rotate = function(r) {
		this.rotated = true;
		cosa = Math.cos(r);
		sina = Math.sin(r);
	}
	
	/**
	* rotateAround(r, x, y);
	*  - rotates the image around (x, y), by 'r' radians.
	**/
	this.rotateAround = function(r, x, y) {
		this.rotatedAround = true;
		cosa = Math.cos(r);
		sina = Math.sin(r);
		a = x; b = y;
	}
	
	/**
	* toDefaultSize();
	*  - returns the zoom, and sizes back to their default values.
	*    width/height rae dependant on the base image size.
	**/
	this.toDefaultSize = function() {
		z = 1;
		img_w = image_w = image.width;
		img_h = image_h = image.height;
	}
	
	/**
	* zoom(z);
	*  - scales the images width and height by 'z' times.
	**/
	this.zoom = function(z) {
		image_w = img_w * z;
		image_h = img_h * z;
	}
		
	/**
	* createSurface();
	*  - wrapper that returns the base image as a surface.
	**/
	this.createSurface = function() {
		return image.createSurface();
	}
	
	/**
	* blit(x, y);
	*  - draws the base image with math assigned. 
	**/
	this.blit = function(x, y) {
		if (this.rotatedAround) rotatearound(x, y, this.mask);
		else if (this.rotated) rotateblit(x, y, image_w, image_h, this.mask);
		else zoomblit(x, y, this.mask);
	}
	
	/**
	* clone();
	*   - returns a clone of this image.
	**/
	this.clone = function() {
		var f = function() { }
		f.prototype = this;
		return new f();
	}
	
	/**
	* internal rotateBlit(x, y, w, h, m);
	*  - draws and rotates the base image at (x, y) with size (w, h) and mask 'm'.
	**/
	var rotateblit = function(x, y, w, h, m) {
		x += (w >>= 1);
		y += (h >>= 1);
		
		// assume origin: (0, 0)
		x1 = x + do_calc_x(-w, -h);
		y1 = y + do_calc_y(-w, -h);
		x2 = x + do_calc_x( w, -h);
		y2 = y + do_calc_y( w, -h);
		x3 = x + do_calc_x( w,  h);
		y3 = y + do_calc_y( w,  h);
		x4 = x + do_calc_x(-w,  h);
		y4 = y + do_calc_y(-w,  h);
		
		image.transformBlitMask(x1, y1, x2, y2, x3, y3, x4, y4, m);
	}
	
	/**
	* internal zoomblit(x, y, m);
	*  - draws the zoomed base image at (x, y) with mask 'm'.
	**/
	var zoomblit = function(x, y, m) {
		image.transformBlitMask(x, y, x + image_w, y, x + image_w, y + image_h, x, y + image_h, m);
	}
	
	/**
	* internal rotatearound(x, y, m);
	*  - draws the base image rotated around point (x, y) with mask 'm'.
	**/
	var rotatearound = function(x, y, m) {
		var xx = x - a;
		var yy = y - b;
		
		// rotate around some point:
		x1 = a + do_calc_x(xx, yy);
		y1 = b + do_calc_y(xx, yy);
		x2 = a + do_calc_x(xx+image_w, yy);
		y2 = b + do_calc_y(xx+image_w, yy);
		x3 = a + do_calc_x(xx+image_w, yy+image_h);
		y3 = b + do_calc_y(xx+image_w, yy+image_h);
		x4 = a + do_calc_x(xx, yy+image_h);
		y4 = b + do_calc_y(xx, yy+image_h);
		
		image.transformBlitMask(x1, y1, x2, y2, x3, y3, x4, y4, m);
	}
	
	// internal:
	var do_calc_x = function(x, y) {
		return (x*cosa)-(y*sina);
	}
	
	// internal:
	var do_calc_y = function(x, y) {
		return (x*sina)+(y*cosa);
	}
}